// Low Frequency Rotary Magnetic Field Exciter Software (Dual 2 plane Helmholtz coils)
// PCB UREBS-06 (MCU MSP430FR5739 16KB FRAM) and aux MSP430FR2433 for monitoring status
// this version is for PEMF 265Hz application
// requires:
//     -55V supply (60V max)
//     -Raspberry pi 5 wired to I2C interface and P2.2 on FR5739


//uses PSPWM (Phase Shift Pulse Width Modulation) method to control output level
//IRS2008 half bridge supports up to 200V on the high side
//combine with 80V/200A MOSFET makes it use able up to 60V(mostly due to 63V electrolytic caps)

// P1.0 , P1.1 test output pads
// P2.5, P2.6 spiral half bridge drivers
// P2.7, P2.4 loop half bridge drivers
// I2C P1.6-SDA and P1.7-SCL (used for interface with Raspberry Pi)
// P2.2 output: request to Pi to get the status and send the setting
// P3.4 to Rpi(to xfer Cal file, P3.5, P4.1 (future use to RPi)
// HB enables, P1.5 and P1.2
// P3.7 burst active pin output to 2nd MCU
// P3.6 connected to 2nd MCU (not used)
// PJ.4 --> XIN, PJ4.5 XOUT
// P2.0(TXD),P2.1(RXD) UART communication to 2nd MCU

// uses 24 MHz external XTAL oscillator for improved stability/accuracy
//DCOCLK   (23.8MHz +/-2% when 0-50degC)
//ACLK    24MHz Max  (>3V)
//SMCLK   24MHz Max  (>3V)
//MCLK    24MHz max (>3V)
//ADC10CLK typ 4.5 MHz (5.5MHz max)
//VLO 8.3KHz typ, 13KHz max

//  ** to enable hex output for command line programming
// Project -> Properties -> MSP430 Hex Utility
// (check the box for Enable MSP430 Hex Utility)
// Click on Output Format Options
// Select TI-TXT from the list.

#include <msp430.h>
#include <stdint.h>
#include <stdbool.h>

#define HBen (BIT2|BIT5)          //two enable bits (on port 1)
#define FBenLoop BIT2             //Loop gate drive en
#define FBenSpiral BIT5           //Spiral gate drive en
#define cM  24                    //24MHz clock rate of MSP430
#define N_CYCLES      13          //50ms worth for burst
// Spiral half-bridge: P2.5 = A (coil), P2.6 = B (caps)
// Loop   half-bridge: P2.4 = A (coil), P2.7 = B (caps)
#define P2MASK_BRIDGES   (BIT4|BIT5|BIT6|BIT7)
#define HBEN        (BIT2|BIT5)  //port 1 chip enables
#define RX_BYTES 71                 //# of UART bytes xfered from 2nd MCU

#define I2C_SLAVE_ADDR 0x71   // (address the Pi uses for I2C)
#define GPIO_NOTIFY    BIT2   // P2.2 used to notify Pi

volatile uint8_t rpiData[236];    //expanded from 80 to 236 to allow cal file transfers. (used for both Tx and Rx)
volatile uint8_t rpiTxSize, rpiRxSize;  //size of xfer for I2C
volatile uint8_t rpiTxIndex = 0;
volatile uint8_t rpiRxIndex = 0;
volatile uint8_t rpiRxDone = 0;

//-----------[ Functions' Prototypes ]--------------
void delay_ms(unsigned int xdms);
void update_Display(void);          //update the LCD display
void readMCU2(void);            //read measured data from MCU2
void calAmpPh(void);            //calibrate amplitude and phase based on measured values

unsigned int x;
unsigned int i, idy;
unsigned int cntMain=1;             //count main loops (for watchdog)
unsigned int cntMainPrev=0;

uint16_t ut16;                      //temporary 16 bit holder
uint8_t enOut=1;                    //ON-OFF setting/status
uint8_t flgBurst=0;                 //flag to indicate start of burst
uint16_t cntHalfCyc=0;              //counter to count bursts
uint8_t cntCyc=0;                   //count cycles
uint8_t cntFullCyc;
uint8_t flgCalMode;                 //flg for calibration mode
unsigned char flgSWmode = 1;        //1 for Level, 2: Burst Off time, 3:Field Mode, 4:Timer mode
uint16_t cntBursts, cntBurstLimit=65000;  //count bursts for length of treatment
uint8_t flgtt=1, ttOn=255, ttOff=0, ttRst=0;       //treatment on and off times
uint8_t flgBurstBusy;
uint8_t ft=3;                       //field type: H,V,R,r
uint8_t  lvlTrg;                    //level target for calibration
uint8_t stOff;
uint8_t cntV=0, cntH=0, cntP=0;     //counters for indicating calibration progress
uint8_t cntVp=0, cntHp=0, cntPp=0;  //counters for indicating count on previous calibration cycle
uint8_t flgCalDone=0;               //flag to indicate
uint8_t calN=0;                     //counter to cycle thru levels 0-5 during calibration


#define wdt_start      (WDTPW + WDTTMSEL + WDTCNTCL + WDTSSEL__VLO + WDTIS__32K)  //  *** WATCHDOG TIMER MODE!!!!!  100us*WDTIS__32K= 3.2secs
//#define wdt_start    (WDTPW + WDTTMSEL + WDTCNTCL + WDTSSEL__VLO + WDTIS__512K)     //  *** WATCHDOG TIMER MODE!!!!!  100us*WDTIS__512K= 51.2secs

#pragma PERSISTENT ( lvl )       // do not save variable so power up state is defaulted to lower
unsigned char lvl = 0;                  //default to lowest power/voltage setting.   //level 0.5mT, 1mT, 2mT, 3mT, 4mT, 5mT
#pragma PERSISTENT ( tOff )
unsigned char tOff = 4;                  //default to 120ms*4 off
#pragma PERSISTENT ( fmode )
char fmode = 'R';                       //default to rotary field (R=rotary, V=Vert, H=Horz)
#pragma PERSISTENT ( tmode )
unsigned char tmode=1;                  //default for timer mode setting

volatile unsigned char rxBuffer[RX_BYTES];
volatile unsigned int  rxIndex = 0;

#define N_CONFIG   3  //  CW, CCW, NoRot
#define N_LVL      6  // 0.5mT, 1mT, 2mT, 3mT, 4mT, 5mT
#define N_SEG     13  // 13 cycles
#pragma PERSISTENT(cfg)
uint8_t cfg=0;  //config  NoRot, CW, CCW

#pragma PERSISTENT(calTrg)
uint8_t calTrg[N_CONFIG][2][N_LVL] = {
    {   // 0: Rotation CW
        {40,50,100,160,200,250},   //Horz 0.5mT, 1mT, 2mT, 3mT, 4mT, 5mT
        {45,70,130,190,220,250}   //VERT 0.5mT, 1mT, 2mT, 3mT, 4mT, 5mT
    },
    {   // 1:Rotation CCW
        {40,55,110,160,200,250},   //HORZ 0.5mT, 1mT, 2mT, 3mT, 4mT, 5mT
        {45,80,140,200,220,250}   //VERT 0.5mT, 1mT, 2mT, 3mT, 4mT, 5mT
    },
    {   // 2: No Rotation
        {40,50,110,160,210,250},   //HORZ 0.5mT, 1mT, 2mT, 3mT, 4mT, 5mT
        {45,70,130,160,210,250}   //VERT 0.5mT, 1mT, 2mT, 3mT, 4mT, 5mT
    }
};

#define MAXCAL    176  // (max ***176: 45,282/256=176.9 )
#pragma PERSISTENT(sc)
//#pragma DATA_SECTION(sc, ".info_defaults_sc")   //234 bytes
uint8_t sc[N_CONFIG][N_LVL][N_SEG] = {
          {   // 0:Horz Rotation CW (max ***176: 45,282/256=176.9 )
              {30,15,10,6,6,6,6,6,6,6,6,6,6},   // 0.5mT
              {50,20,20,15,15,15,15,15,15,15,15,15,15},   // 1mT
              {75,40,20,20,20,20,20,20,20,20,20,20,20},   // 2mT
              {100,50,35,25,25,25,25,25,25,25,25,25,25},   // 3mT
              {176,100,60,50,40,40,40,40,40,40,40,40,40},   // 4mT
              {176,160,90,70,50,50,50,50,50,50,50,50,50}    // 5mT  {176,176,176,176,100,75,75,75,75,75,75,75,75}
          },
          {   // 1:Horz Rotation CCW
              {40,20,15,7,7,7,7,7,7,7,7,7,7},   // 0.5mT
              {50,20,20,15,15,15,15,15,15,15,15,15,15},   // 1mT
              {100,50,20,20,20,20,20,20,20,20,20,20,20},   // 2mT
              {176,100,50,40,40,40,40,40,40,40,40,40,40},   // 3mT
              {176,176,100,60,60,60,60,60,60,60,60,60,60},   // 4mT
              {176,176,150,100,80,80,80,80,80,80,80,80,80}    // 5mT
          },
          {   // 2:Horz No Rotation
              {35,20,10,7,7,7,7,7,7,7,7,7,7},   // 0.5mT
              {50,20,20,15,15,15,15,15,15,15,15,15,15},   // 1mT
              {100,50,25,25,25,25,25,25,25,25,25,25,25},   // 2mT
              {150,75,50,35,35,35,35,35,35,35,35,35,35},   // 3mT
              {176,125,75,50,50,50,50,50,50,50,50,50,50},   // 4mT
              {176,150,100,75,75,75,75,75,75,75,75,75,75}    // 5mT
          }
      };



#pragma PERSISTENT(lc)
//#pragma DATA_SECTION(lc, ".info_defaults_lc") //234bytes
uint8_t lc[N_CONFIG][N_LVL][N_SEG] = {
          {   // 0:Vert Rotation CW (max ***176: 45,282/256=176.9 )
              {20,10,4,4,4,4,4,4,4,4,4,4,4},   // 0.5mT
              {25,15,7,7,7,7,7,7,7,7,7,7,7},   // 1mT
              {50,25,12,12,12,12,12,12,12,12,12,12,12},   // 2mT
              {90,60,40,20,20,20,20,20,20,20,20,20,20},   // 3mT
              {176,100,50,30,30,30,30,30,30,30,30,30,30},   // 4mT
              {176,150,90,50,40,40,40,40,40,40,40,40,40}    // 5mT   {176,176,176,100,50,50,50,50,50,50,50,50,50}
          },
          {   // 1:Vert Rotation CCW
              {20,10,4,4,4,4,4,4,4,4,4,4,4},   // 0.5mT
              {20,10,5,5,5,5,5,5,5,5,5,5,5},   // 1mT
              {50,25,10,10,10,10,10,10,10,10,10,10,10},   // 2mT
              {176,100,50,20,20,20,20,20,20,20,20,20,20},   // 3mT
              {176,140,70,30,30,30,30,30,30,30,30,30,30},   // 4mT
              {176,160,100,80,40,40,40,40,40,40,40,40,40}    // 5mT
          },
          {   // 2:Vert No Rotation
              {20,10,4,4,4,4,4,4,4,4,4,4,4},   // 0.5mT
              {30,15,6,6,6,6,6,6,6,6,6,6,6},   // 1mT
              {60,30,15,10,10,10,10,10,10,10,10,10,10},   // 2mT
              {176,80,50,20,20,20,20,20,20,20,20,20,20},   // 3mT
              {176,120,50,30,30,30,30,30,30,30,30,30,30},   // 4mT
              {176,176,125,70,40,40,40,40,40,40,40,40,40}    // 5mT
          },
      };


//#pragma PERSISTENT(ph)
#pragma DATA_SECTION(ph, ".infoA")
//# goes into TB0CCR0 where 88 is the ideal 90 deg setting.
// use 90<<8 == 23040 (close to 90) max number is 45282/256=176
uint8_t ph[N_CONFIG][N_LVL][4] = {
    {   //0: Phase Rotation CW
        {80,80,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 0.5mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 1mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 2mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 3mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 4mT
        {90,90,90,90} //,90,90,90, 90,90,90,90,90,90}    // 5mT
    },
    {   // 1: Phase Rotation CCW
        {90,90,90,90}, //,90,110,130, 130,130,130,130,130,130},   // 0.5mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 1mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 2mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 3mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 4mT
        {70,90,90,90} //,90,90,90, 90,90,90,90,90,90}    // 5mT
    },
    {   // 2: No Rotation (not used for no rotation)
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 0.5mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 1mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 2mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 3mT
        {90,90,90,90}, //,90,90,90, 90,90,90,90,90,90},   // 4mT
        {90,90,90,90} //,90,90,90, 90,90,90,90,90,90}    // 5mT
    }
};
//*********************************************************************************
//******************************* MAIN ********************************************
int main(void)
{
    WDTCTL = WDTPW | WDTHOLD;   // stop watchdog timer

    // FRAM wait states for the FR5739 are inserted automatically
    //configure for 24 MHz external oscillator PJ.4(XIN), PJ.5(XOUT)
    PJSEL0 |=  BIT5 | BIT4;     //set PJ.5 as xout and PJ.4 as xin
    CSCTL0_H = 0xA5;                            //password
    CSCTL1 |= DCORSEL + DCOFSEL0 + DCOFSEL1;   // Set max. DCO setting
    CSCTL2 = SELA_0 + SELS_0 + SELM_0;        // set ACLK = MCLK = DCO : set all clk sources for XT1
    CSCTL3 = DIVA_0 + DIVS_0 + DIVM_0;        // set all dividers to 0
    CSCTL4 |= XT1DRIVE_3 + XTS + XT1BYPASS;   //highest drive, high freq mode, external clk
    //wait for clock to start
    //clear XT1 and DCO fault flag and interrupt
    do{CSCTL5 &= ~XT1OFFG; SFRIFG1 &= ~OFIFG;}
    while (SFRIFG1 & OFIFG);

    // PORT 1  --  7(I2C-SCL) 6(I2C-SDA)
    P1DIR = HBen+BIT0;     //P1.2,P1.5 en outputs for HB, all else inputs
    // Configure I2C pins
    P1SEL1 |= BIT6 | BIT7;  // P1.6 = SDA, P1.7 = SCL


    // PORT 2 (all outputs)
    // 7,6,5,4 half bridge driver outputs)  2(LCD reset) 1(sw) 0(sw)
    P2DIR = BIT7 | BIT6 | BIT5 | BIT4 | BIT2;   //outputs
    P2OUT = BIT7 | BIT6 | BIT5 | BIT4 ;          //high
    P2SEL1 |= BIT0 | BIT1;                      // UCA0TXD=P2.0, UCA0RXD=P2.1  (UART)
    P2SEL0 &= ~(BIT0 | BIT1);
    // Configure GPIO for notify
    P2DIR |= GPIO_NOTIFY;   // P1.4 output
    P2OUT &= ~GPIO_NOTIFY;  // Low at start

    // PORT 3 (all outputs)
    P3DIR = BIT7 | BIT6 | BIT5| BIT4 | BIT3 | BIT2 | BIT1 | BIT0; //all output P3.4,5 go to drive RPi interrupts
    P3OUT = 0x00;   //

    //PORT4
    P4DIR = 0x00 | BIT1;    //all inputs except BIT1 which is used to future RPi interrupt use
    P4OUT = 0x00;   // BIT1 defaults low

    // Configure ADC10
    // By default, REFMSTR=1 => REFCTL is used to configure the internal reference
     while(REFCTL0 & REFGENBUSY);              // If ref generator busy, WAIT
     REFCTL0 |= REFVSEL_2+REFON;               // Select internal ref = 2.5V. Internal Reference ON
     ADC10CTL0 = ADC10SHT_2 + ADC10ON;
     ADC10CTL1 |= ADC10SHP;                    // ADCCLK = MODOSC; sampling timer
     ADC10CTL2 |= ADC10RES;                    // 10-bit conversion results
     //ADC10IE |= ADC10IE0;                    // Enable ADC conv complete interrupt (routine now uses polling)
     ADC10MCTL0 |= ADC10INCH_1 + ADC10SREF_1;  // A0 ADC input select; Vref=1.5V

    // Configure
    PMMCTL0_H = PMMPW_H;                        // Unlock the PMM registers
    PM5CTL0 &= ~LOCKLPM5;                       // Disable the GPIO power-on default high-impedance mode to activate previously configured port settings

    delay_ms(1000);         //3000ms startup delay for power supply power up and cap charging
    P1OUT &= ~HBen;         //Half Bridge OFF

   //setup hardware clock resonant frequency, output goes to P1.0 (scanning starting point)
    TA0CCR0 = (24000000/530)-1;    // PWM Period
    //TA0CCTL1 = OUTMOD_7;                      // CCR1 reset/set
    //TA0CCR1 = TA0CCR0>>1;                     // CCR1 PWM duty cycle (50% is divid by 2)
    TA0CCR1 = 10000;                     // CCR1 PWM duty cycle  1-45,282
    TA0CCTL0 = CCIE;       // Enable interrupts for CCR0 and CCR1
    TA0CCTL1 = CCIE;
    //TA0CTL = TASSEL__SMCLK | MC__CONTINUOUS | TACLR;  // SMCLK, up mode, clear TAR
    TA0CTL = TASSEL__SMCLK | MC__UP | TACLR;     // SMCLK, Up mode, clear TAR

    //setup interrupt to control the LOOP timing
    TA1CCR0 = (24000000/530)-1;    // PWM Period
    TA1CCR1 = 4000;       // CCR1 PWM duty cycle  1-45282
    TA1CCTL0 = CCIE;       // Enable interrupts for CCR0 and CCR1
    TA1CCTL1 = CCIE;
    TA1CTL = TASSEL__SMCLK | MC__UP | TACLR;     // SMCLK, Up mode, clear TAR

    // I2C Slave mode
    UCB0CTLW0 = UCSWRST;        // Put eUSCI_B0 into reset
    UCB0CTLW0 |= UCMODE_3 | UCSYNC;   // I2C slave, synchronous
    UCB0I2COA0 = I2C_SLAVE_ADDR | UCOAEN;  // Own address enable
    UCB0CTLW0 &= ~UCSWRST;           // Release for operation
    UCB0IE |= UCTXIE0 | UCRXIE0;    // Enable TX and RX interrupt



    delay_ms(5000);         //5 second startup delay to charge caps

   //setup timer phase adjustment
    TB0CCTL0 = CCIE;                          // TACCR0 interrupt enabled
    TB0CCR0 = 1000;
    TB0CTL = TASSEL__SMCLK |ID__8 | MC__UP | TBCLR;     // 24MHz, up mode

    // UART SETUP for 112000 baud @24MHz SMCLK
    UCA0CTLW0 = UCSWRST;
    UCA0CTLW0 |= UCSSEL__SMCLK;   // SMCLK = 24 MHz
    UCA0BRW   = 13;                        // Integer divider
    UCA0MCTLW = UCOS16 | (6 << 4) | (0 << 8);
    // UCOS16 = oversampling
    // UCBRF = 6 (fractional part ~0.39)
    // UCBRS = 0 (no extra correction)

    UCA0CTLW0 &= ~UCSWRST;
    UCA0IE    |= UCRXIE;                   // Enable RX interrupt

    __enable_interrupt();
    SFRIE1 |= WDTIE;   // Enable WDT interrupt
//    WDTCTL = wdt_start; // start the Watchdog


    // **************************** MAIN LOOP **********************************
 while(1){
    for (cntBursts=0; cntBursts<cntBurstLimit; cntBursts++) { //count bursts for length of treatment

        // spiral coil half bridge use P2.5(coil), and P2.6(caps)
        // loop coil half bridge uses P2.4(coil), and P2.7(caps)
        //265Hz(3774us) 50% is 1.8868ms
        //set cntBurstLimit
        if (flgtt==1) { //treatment on
            cntBurstLimit=ttOn<<8;
        } else {        //treatment off
            if (ttOff>0){
                cntBurstLimit=ttOff<<8;
            } else if (ttOff==255){
                //should stay off after ttOn is done
                cntBurstLimit=65000;
                if (cntBursts>60000) cntBursts=10;  //reset to never reach limit
            } else {
                cntBurstLimit=1; //just a minimun amount
            }
        }

        //determine field Type required
        switch(fmode) {
            case 'V': ft=1; break; //vertical only
            case 'H': ft=2; break; //Horz only
            case 'R': ft=3; break; //Rotary spiral leads
            case 'r': ft=4; break; //Rotary loop leads
            case '1': ft++; if (ft>2) ft=1; break;    //toggles between V and H
            case '2': ft++; if (ft>4) ft=3; break;   //toggles rotation direction;
            case '3': ft++; if (ft>4) ft=1; break;   // sequence thru all
            default:ft=3; break; //Rotary spiral leads
        }//sw
        switch(ft) {
            case 1: cfg=2; break;  //no rotation
            case 2: cfg=2; break;  //no rotation
            case 3: cfg=0; break; //rotation
            case 4: cfg=1; break; //reverse rotation
            default: cfg=0; break;
        }//sw

        //discharge the capacitors before the next burst.
        __disable_interrupt();
        P2OUT &= ~(BIT5 | BIT6 | BIT4 | BIT7);    //half bridges to gnd
        P1OUT |= HBen;     //full bridge chips enabled
        delay_ms(50);
        P1OUT &= ~HBen;     //half bridge chips disabled

        P1OUT |= BIT0;  //scope trigger output
        __delay_cycles(cM*10);  //10us for triggering
        P1OUT &= ~BIT0;  // return low

        __enable_interrupt();
        if (enOut&&flgtt) {  //test for on/off setting
            flgBurst=1;  //flg to start real burst
            while (flgBurstBusy==0);  //wait for burst to start
            P3OUT |= BIT7; //set bit7 high to indicate burst started
            while (flgBurstBusy); //wait for burst to be don;
            P1OUT &= ~HBen;     //half bridge chips disabled
        }

        //burst is done, get the bytes from the monitor MCU
        readMCU2();  //get the updated measured data

        cntMain++;              //increment in order to verify program working.

        // adjust amplitude and phase based on measured values
        if (flgCalMode>0){
            switch(flgCalMode) {
                case 1: cntBursts=0;  tOff=4; flgCalMode=2; break; //initial setup
                case 2: fmode='V'; lvl=calN; break;    //Vert cal all levels
                case 3: fmode='H'; lvl=calN; break;    //HORZ cal all levels
                case 4: fmode='R'; lvl=calN; break;    //R cal all levels
                case 5: fmode='r'; lvl=calN; break;    //R cal all levels
                default: fmode='V'; lvl=calN; break;   //Vert cal all levels
            }//sw
            calAmpPh();  //adjust calibration table based on measured values
            if ((cntBursts>10)&&(cntH==cntHp)&&(cntV==cntVp)&&(cntP==cntPp)){
                cntH=0; cntV=0; cntP=0;
                if (calN<5) {   //0-5 levels
                    calN++; cntBursts=0;    //advance to next level calibration
                } else {
                    calN=0; flgCalMode++; cntBursts=0;  //restart at level 0 and next mode
                }
            }
            if (flgCalMode>5) flgCalMode=0;  //mark as done all calibrations
        }//if flgCalMode

        // do delay in 120ms segments so keypad can be checked, and display updated
      for (idy=1; idy<=tOff; idy++) {
          delay_ms(120);
          update_Display();
      }//for loop for off delay

  }//for cntBursts
    if (flgtt==1) {
        flgtt=0;  //treatment on, so turn off
    } else {
        flgtt=1;  //last cycle was treatment off, so turn on
    }
}//while

    //return 0;
}//main




//********************* Interrupt Routines ********************************
// ***************************************************************


// **********************************************************
// *******************   WDT INTERRUPT **********************
// Watchdog Timer interrupt service routine
// arrives here when HB needs to be sent.
//  Note that VLO(not very stable) is used for watchdog so that the HBs are not aligned in time between different units
#pragma vector=WDT_VECTOR
__interrupt void WDT_ISR(void)
{
    //verify program working by checking cntMain is incrementing
    if (cntMain==cntMainPrev) {
        //MCU hung, so MCU needs reboot
        PMMCTL0 |= PMMSWBOR;    //cause a software brown out reset
    } else {
        cntMainPrev=cntMain;
    }
}//WD interrupt

//****************** delay_ms ******************
void delay_ms(unsigned int xdms)
{
    unsigned int xi;
    for (xi = xdms; xi > 0; xi--) {
        __delay_cycles(cM*1000);   //1ms delay
    }
}

// ***************** TIMER0-A ISR SPIRAL resonance***************************
#pragma vector = TIMER0_A0_VECTOR
__interrupt void TIMER0_A0_ISR(void)
{
    if ((flgBurst)&(flgBurstBusy==0)) {
        //indicates request to start the burst
        P2OUT &= ~(BIT5 | BIT6);    //spiral
        if ((cfg==0)||(cfg==2)) {
            P2OUT |= (BIT4 | BIT7);     //loop: setting for spiral leads
            //set the phase between resonances
            ut16 = ((uint16_t)ph[cfg][lvl][0]) << 8; //stored value *256 (90*256=23040)
            TA1R = ut16; // 90 deg is ideal 88.4 is 22641
            TA0R = 0;   //set the phases between them
            // Timer0_A
            TA0CTL   &= ~TAIFG;
            TA0CCTL0 &= ~CCIFG;   // not strictly needed, CCR0 auto-clears on ISR entry
            TA0CCTL1 &= ~CCIFG;
            TA0CCTL2 &= ~CCIFG;

            // Timer1_A
            TA1CTL   &= ~TAIFG;
            TA1CCTL0 &= ~CCIFG;
            TA1CCTL1 &= ~CCIFG;
            TA1CCTL2 &= ~CCIFG;
        }
        if (cfg==1) {
            P2OUT &= ~(BIT4 | BIT7);     //loop: setting for loop leads
            //set the phase between resonances
            ut16 = ((uint16_t)ph[cfg][lvl][0]) << 8; //stored value *256 (90*256=23040)
            TA1R = ut16; // 90 deg is deal 88.4 is 22641
            TA0R = 0; //set the phases between them
            // Timer0_A
            TA0CTL   &= ~TAIFG;
            TA0CCTL0 &= ~CCIFG;   // not strictly needed, CCR0 auto-clears on ISR entry
            TA0CCTL1 &= ~CCIFG;
            TA0CCTL2 &= ~CCIFG;

            // Timer1_A
            TA1CTL   &= ~TAIFG;
            TA1CCTL0 &= ~CCIFG;
            TA1CCTL1 &= ~CCIFG;
            TA1CCTL2 &= ~CCIFG;
        }
        cntCyc=1;
        flgBurst=0;
        flgBurstBusy=1;   //burst been performed
        cntHalfCyc=0;
        if (cntBursts>0){
            //skips 1st burst to avoid excessive level
            //determine field type and hence which full bridges are enabled
            switch(ft) {
                case 1: P1OUT |= FBenLoop; break;     //full bridge chip enabled
                case 2: P1OUT |= FBenSpiral; break;     //full bridge chip enabled
                case 3: P1OUT |= HBen; break;     //full bridge chips enabled
                case 4: P1OUT |= HBen; break;     //full bridge chips enabled
                default: P1OUT |= HBen; break;    //full bridge chips enabled
            }//sw
        }//if cntBursts
    }

    P2OUT ^= BIT6;  //toggles every interrupt

    //amplitude calibration
    if (cntHalfCyc<=N_CYCLES*2) {
        //cal amplitude
        cntFullCyc=cntHalfCyc>>1;  //divid by 2
        ut16 = (uint16_t)sc[cfg][lvl][cntFullCyc]<<8;  //*256, set the duty cycle that determines the magnetic field strength
        TA0CCR1 = ut16;
        if (cntFullCyc==7){
            ut16--; //breakpoint for troubleshooting
        }
    }

 /*
    //phase calibration
    if ((cntHalfCyc>=1)&(cntHalfCyc<=20)) {
        //cal phase but only every cycle (not half cycle)
        if (P2OUT&BIT6) {
            //does every other half cycle, because can only adjust phase every cycle
            cntFullCyc=cntHalfCyc>>1;  //divid by 2
            ut16 = (ph[cfg][lvl][cntFullCyc]) << 8; //stored value *256 (90*256=23040)
            TA1R = ut16; // 90 deg is ideal 88.4 is 22641
            TA0R = 0; //set the phases between them
        }
    }
*/
    if (cntHalfCyc>=(N_CYCLES*2)) {
        flgBurstBusy=0;     //set flag for burst done
    }

    cntHalfCyc++;     //increment burst counter
    cntCyc++;
    if (cntCyc>=5) cntCyc=1;  //reset
}

#pragma vector = TIMER0_A1_VECTOR
__interrupt void TIMER0_A1_ISR(void)
{
    switch (__even_in_range(TA0IV, TA0IV_TAIFG)) {
        case TA0IV_NONE:   break;                   // No interrupt
        case TA0IV_TACCR1: P2OUT ^= BIT5; break;    // Duty point toggle
        case TA0IV_TACCR2: break;                   // Handle CCR2 if needed
        case TA0IV_TAIFG:  break;                   // Handle overflow if needed
        default: break;
    }
}

// ****************** Timer 1 for LOOP resonance **************
// TA1 CCR0 ISR: period boundary (toggle loop B)
#pragma vector = TIMER1_A0_VECTOR
__interrupt void TIMER1_A0_ISR(void)
{
    P2OUT ^= BIT4;  //Period boundary, toggle
    if (P2OUT&BIT4){  //test rising edge of bit4
        P2OUT &= ~BIT7;  //ensure low
    }else {  //falling edge
        P2OUT |= BIT7;  //ensure high
    }

    if (cntHalfCyc<=N_CYCLES*2) {
        cntFullCyc=cntHalfCyc>>1;  //divid by 2
        ut16 = (uint16_t)lc[cfg][lvl][cntFullCyc]<<8; //*256, set the duty cycle that determines the magnetic field strength
        TA1CCR1 = ut16;
    }
}

// TA1 CCR1+overflow ISR: handle CCR1 (toggle loop A)
#pragma vector = TIMER1_A1_VECTOR
__interrupt void TIMER1_A1_ISR(void)
{
    switch (__even_in_range(TA1IV, TA1IV_TAIFG)) {
        case TA1IV_NONE:   break;         // No interrupt
        case TA1IV_TACCR1:
            P2OUT ^= BIT7;
            if (P2OUT&BIT7){  //rising edge of bit7
                P2OUT |= BIT7;  //ensure high
            } else {  //falling edge of bit7
                P2OUT &= ~BIT7; // ensure low
            }
            break;  // Duty boundary toggle
        case TA1IV_TACCR2: break;         // Handle CCR2 if needed
        case TA1IV_TAIFG:  break;         // Handle overflow if needed
        default: break;
    }
}



// ***************** TIMER B0 INTERRUPT ***************************
// Timer B0 interrupt service routine
#pragma vector = TIMER0_B0_VECTOR
__interrupt void Timer_B (void)
{
    __bic_SR_register_on_exit(LPM0_bits); // wake main loop
}


// *******************  UART **********************
#pragma vector=USCI_A0_VECTOR
__interrupt void USCI_A0_ISR(void) {
    switch (__even_in_range(UCA0IV, USCI_UART_UCTXCPTIFG)) {
    case USCI_NONE: break;
    case USCI_UART_UCRXIFG:
        if (rxIndex < RX_BYTES) {
            rxBuffer[rxIndex++] = UCA0RXBUF;
        }
        break;
    default: break;
    }
}

// *******************  I2C **********************
#pragma vector=USCI_B0_VECTOR
__interrupt void USCI_B0_ISR(void)
{
    switch(__even_in_range(UCB0IV, USCI_I2C_UCBIT9IFG))
    {
        case USCI_NONE:
            break;

        case USCI_I2C_UCRXIFG0:  // received a byte from Pi
            if (rpiRxIndex < rpiRxSize) {
                rpiData[rpiRxIndex] = UCB0RXBUF;
                rpiRxIndex++;
                if (rpiRxIndex >= rpiRxSize) {
                    rpiRxDone = 1;
                    rpiRxIndex = 0;
                }
            }
            break;

        case USCI_I2C_UCTXIFG0:        // TX buffer ready
            if (rpiTxIndex < rpiTxSize) {
                UCB0TXBUF = rpiData[rpiTxIndex];
                rpiTxIndex++;
            } else {
                UCB0TXBUF = 0xFF;      // send 0xFF if master over-reads
            }
            break;
        case USCI_I2C_UCSTPIFG:        // STOP detected (not always triggered on RPi)
            UCB0IFG &= ~UCSTPIFG;      // clear STOP flag
            break;
        default:
            break;
    }
}

//****************************************************************************
// ***************************** Subroutines *******************************


//update the display (meaning info sent to Raspberry Pi)
//also gets updated settings from the Raspberry Pi
void update_Display(void)
{
    uint8_t lx, idx, ncfg, nlvl, nseg;

    // this routine now builds a rpiTx[]
    rpiData[0]=enOut;
    rpiData[1]=lvl;  //0-5
    rpiData[2]=fmode; //char
    rpiData[3]=tOff;  //burst off time
    rpiData[4]=ttOn;  //treatment on time
    rpiData[5]=ttOff;  //treatment off time
    rpiData[6]=ttRst;  //treatment restart
    rpiData[7]=flgCalMode;  //calibration mode
    rpiData[8]=calN;        //calibration level
    rpiData[9]=ph[cfg][lvl][0];        //phase calibration

    for (idx=0; idx<=19; idx++) {  //Spiral Voltages
        rpiData[20+idx]=rxBuffer[31+idx];
    }
    for (idx=0; idx<=19; idx++) {  //Loop Voltages
        rpiData[40+idx]=rxBuffer[51+idx];
    }
    for (idx=0; idx<=19; idx++) {  //Phases
        rpiData[60+idx]=rxBuffer[11+idx];
    }

    UCB0TXBUF = rpiData[0];
    rpiTxSize=80;  rpiRxSize=20;  //specify size for Xfer
    rpiTxIndex=0; rpiRxIndex = 0;  rpiRxDone = 0;   //initialize all the indexes and flag
    P2OUT |= GPIO_NOTIFY;   // Set high (tell Pi to read)
    for (lx=1; lx<=40; lx++){ //waits for tx done (0-79) (~9ms) plus the 20 RX, plus the pi interrupt time
        delay_ms(1);
        if (rpiRxDone==1) break;
    }
    //at this point the rpiData array has been updated with the sets.csv from the RPi
    P2OUT &= ~GPIO_NOTIFY;  // Clear notify

    //next check for changes in rpiData
    // when the change is acknowledged it gets updated in rpiRxDataAck
  if ((rpiData[0]&0xF0)==0xA0){
    for (idx=1; idx<=19; idx++){
        if ((rpiData[0]&0x0F)!=enOut){  // ON/OFF setting (lower nibble only
            enOut=(rpiData[0]&0x0F);
        }
        //disable ability to change settings if in calibration mode
        if (flgCalMode==0)  { //only allow changes when not in calibration mode
            if (rpiData[1]!=lvl){  //received a change in the settings field level
                lvl=rpiData[1];
                if (lvl>5) lvl=0;  //restrict range
            }
            if (rpiData[2]!=fmode){  //received a change in the settings field level
                fmode=rpiData[2];  //update
                //verify input
                if (fmode!='V' && fmode!='H' && fmode!='R' && fmode!='r' &&
                    fmode!='1' && fmode!='2' && fmode!='3') {
                    fmode = 'R';   // default
                }
            }
            if (rpiData[3]!=tOff){  //received a change in the settings of toff
                tOff=rpiData[3];
                if (tOff<1) tOff=1;  //set min value
            }
            if (rpiData[4]!=ttOn){  //treatment on time
                ttOn=rpiData[4];
                if (ttOn==0) ttOn=1;  //set min value
            }
            if (rpiData[5]!=ttOff){  //treatment off time
                ttOff=rpiData[5];
            }
            if (rpiData[6]!=ttRst){  //treatment Restart
                ttRst=rpiData[6];
                flgtt=1;  //start with on treatment and reset cntBursts
                cntBursts=0;
            }
            if (rpiData[7]!=flgCalMode){  //received a request to initialize calibration routine
                flgCalMode=rpiData[7];   //sets the flag to start the calibration
            }
            if (rpiData[8]!=0){  //1= send cal file to RPi, 128 Rcv Cal file from RPi
                //NOTE the reuse of rpiData array will wipe out the sets info but no longer it.
                if (rpiData[8]==1) {
                    //---------------sc--------------
                    //Spiral cal file 3*6*13=234, fill the txbuffer array
                    idx=0;  //initialize
                    for (ncfg=0; ncfg<=2; ncfg++) {
                        for (nlvl=0; nlvl<=5; nlvl++) {
                            for (nseg=0; nseg<=12; nseg++) {
                                rpiData[idx]=sc[ncfg][nlvl][nseg];
                                idx++;  //manually increment
                            }
                        }
                    }
                    //flag request to send the calibration files to the RPi
                    rpiTxSize=234;  //I2C Xfer size
                    rpiTxIndex=0;  //reset to beginning of array (used in I2C ISR)
                    P3OUT |= BIT4;  //set P3.4 high to cause interrupt in RPI
                    //wait for rpiTxIndex to be 233
                    //while (rpiTxIndex!=233);  //
                    for (lx=1; lx<=40; lx++){ //waits for tx done (~22ms)
                        delay_ms(1);
                        if (rpiTxIndex>=233) break;
                    }
                    //-------------lc----------------
                    //load the lc values
                    rpiTxSize=234;  //I2C Xfer size
                    rpiTxIndex=0;  //reset to beginning of array (used in I2C ISR)
                    //LOOP cal file 3*6*13=234, fill the txbuffer array
                    idx=0;  //initialize
                    for (ncfg=0; ncfg<=2; ncfg++) {
                        for (nlvl=0; nlvl<=5; nlvl++) {
                            for (nseg=0; nseg<=12; nseg++) {
                                rpiData[idx]=lc[ncfg][nlvl][nseg];
                                idx++;  //manually increment
                            }
                        }
                    }
                    //wait till Xfer done
                    for (lx=1; lx<=40; lx++){ //waits for tx done (~22ms)
                        delay_ms(1);
                        if (rpiTxIndex>=233) break;
                    }
                    //-------------ph----------------
                    //load the ph values
                    rpiTxSize=72;  //I2C Xfer size
                    rpiTxIndex=0;  //reset to beginning of array (used in I2C ISR)
                    //ph cal file 3*6*4=72, fill the txbuffer array
                    idx=0;  //initialize
                    for (ncfg=0; ncfg<=2; ncfg++) {
                        for (nlvl=0; nlvl<=5; nlvl++) {
                            for (nseg=0; nseg<=3; nseg++) {
                                rpiData[idx]=ph[ncfg][nlvl][nseg];
                                idx++;  //manually increment
                            }
                        }
                    }
                    //wait till Xfer done
                    for (lx=1; lx<=40; lx++){ //waits for tx done (~22ms)
                        delay_ms(1);
                        if (rpiTxIndex>=71) break;
                    }

                    P3OUT &= ~BIT4;  //set P3.4 low
                }
                //-----------------------------------------------------------------------------------------------
                //check for the action of loading the cal data from the RPi
                if (rpiData[8]==0x80) {
                    rpiRxSize=234; rpiRxIndex = 0;  rpiRxDone = 0;   //initialize all the indexes and flag
                    idx = UCB0RXBUF;  //ensure the I2C interrupt flag is cleared (reading the register clears the flag)
                    //requires GPIO27 on RPi to go high for interrupt.
                    //-------------sc----------------
                    //step 1 get I2C data byte by byte and update the Spiral cal file 3*6*13=234
                    P3OUT |= BIT5;  //set P3.5 high to cause interrupt in RPI
                    for (lx=1; lx<=100; lx++){ //waits for rx done (0-233) (max 100ms wait time)
                        delay_ms(1);
                        if (rpiRxDone==1) break;
                    }
                    if (rpiRxDone==1) {
                        //copy the data from the rpiData array to the sc calibration array
                        idx=0;  //initialize
                        for (ncfg=0; ncfg<=2; ncfg++) {
                            for (nlvl=0; nlvl<=5; nlvl++) {
                                for (nseg=0; nseg<=12; nseg++) {
                                    sc[ncfg][nlvl][nseg]= rpiData[idx];  //save the data to the cal array
                                    idx++;  //manually increment
                                }
                            }
                        }
                    }
                    //-------------lc----------------
                    //step 2 get and write the Loop cal file 3*6*13=234
                    rpiRxSize=234; rpiRxIndex = 0;  rpiRxDone = 0;   //initialize all the indexes and flag
                    for (lx=1; lx<=100; lx++){ //waits forrx done (0-233) (max 100ms wait time)
                        delay_ms(1);
                        if (rpiRxDone==1) break;
                    }
                    if (rpiRxDone==1) {
                        //copy the data from the rpiData array to the sc calibration array
                        idx=0;  //initialize
                        for (ncfg=0; ncfg<=2; ncfg++) {
                            for (nlvl=0; nlvl<=5; nlvl++) {
                                for (nseg=0; nseg<=12; nseg++) {
                                    lc[ncfg][nlvl][nseg]= rpiData[idx];  //save the data to the cal array
                                    idx++;  //manually increment
                                }
                            }
                        }
                    }
                    //-------------ph----------------
                    //step 3 get and write the Phase cal file 3*6*4=72
                    rpiRxSize=72; rpiRxIndex = 0;  rpiRxDone = 0;   //initialize all the indexes and flag
                    for (lx=1; lx<=100; lx++){ //waits for rx done (0-71) (max 100ms wait time)
                        delay_ms(1);
                        if (rpiRxDone==1) break;
                    }
                    if (rpiRxDone==1) {
                        //copy the data from the rpiData array to the ph calibration array
                        idx=0;  //initialize
                        for (ncfg=0; ncfg<=2; ncfg++) {
                            for (nlvl=0; nlvl<=5; nlvl++) {
                                for (nseg=0; nseg<=3; nseg++) {
                                    ph[ncfg][nlvl][nseg]= rpiData[idx];  //save the data to the cal array
                                    idx++;  //manually increment
                                }
                            }
                        }
                    }
                    //-------------calTrg----------------
                    //step 4 get and write the calTrg[N_CONFIG][2][N_LVL] file 3*2*6=36
                    rpiRxSize=36; rpiRxIndex = 0;  rpiRxDone = 0;   //initialize all the indexes and flag
                    for (lx=1; lx<=100; lx++){ //waits for rx done (0-71) (max 100ms wait time)
                        delay_ms(1);
                        if (rpiRxDone==1) break;
                    }
                    if (rpiRxDone==1) {
                        //copy the data from the rpiData array to the ph calibration array
                        idx=0;  //initialize
                        for (ncfg=0; ncfg<=2; ncfg++) {
                            for (nseg=0; nseg<=1; nseg++) { //0-1: (H/V) not really segment but reusing variable
                                for (nlvl=0; nlvl<=5; nlvl++) {
                                    calTrg[ncfg][nseg][nlvl]= rpiData[idx];  //save the data to the cal array
                                        idx++;  //manually increment
                                }
                            }
                        }
                    }


                    P3OUT &= ~BIT5;  //set P3.5 low
                }

            }
        }//if cal mode
    }//for
  }//validity test

}




//---------------------------------------------------
//get the updated data from MCU2
void readMCU2(void)
{
    //burst is done, get the bytes from the monitor MCU
    for (i=0; i<RX_BYTES; i++) rxBuffer[i]=0;  //clear array
    P3OUT &= ~BIT7;  //set bit7 low (triggers the sending of the bytes)
    delay_ms(20); //time to send 61 bytes at 56000 baud is 11ms
   //all bytes received in the background interrrupts.
    if (rxIndex >= RX_BYTES) {
        rxIndex = 0;          // ready for next packet
    }
}


//---------------------------------------------------------------
// calibrate amplitude and phase based on measured values
// uses rxBuffer as the source of the measured values
// calibration arrays are sc[cfg][lvl][burst#], lc[][][], and ph[][][]
void calAmpPh(void)
{
    //update the level based on measured values (rxBuffer)
    //11-30 : phase measurements(20 max)
    //31-50: detector spiral (20 max)
    //51-70: detector loop (20 max)


    cntHp=cntH; cntVp=cntV; cntPp=cntP;  //assign values to previous values

    //only do calibration for constant modes (not alternating modes)
    if ((fmode=='V')||(fmode=='H')||(fmode=='R')||(fmode=='r')) {

        //HORZ calibration------------------
        //skip if ft=1
        if ((ft!=1)&&(rxBuffer[40]!=0))  {
            lvlTrg= calTrg[cfg][0][lvl]; //set horz level target
            //find starting point of 1st valid measurement by looking for last valid measurement
            if (rxBuffer[44]==0) stOff=31;
            else if (rxBuffer[45]==0) stOff=32;
            else if (rxBuffer[46]==0) stOff=33;
            else if (rxBuffer[47]==0) stOff=34;
            else {stOff=35;}
            stOff++;

            for (i=0; i<=12; i++) {  //scan the 13 burst measurements
                if (rxBuffer[stOff+i]<lvlTrg) {
                    //allow increase only if following 3 cycles are not too high
                    if (i<=8){
                        if ((rxBuffer[stOff+i]<lvlTrg)&&(rxBuffer[stOff+i+1]<lvlTrg)&&(rxBuffer[stOff+i+2]<lvlTrg)) {
                            if (i==0){
                                if (sc[cfg][lvl][i]<(MAXCAL-2)) {
                                    sc[cfg][lvl][i]++; sc[cfg][lvl][i]++;  //checks value before increment
                                    sc[cfg][lvl][i+1]++;  cntH++;  //do 2nd one slightly
                                    break;  //if incremented breaks loop and waits till next burst
                                }
                            } else if (sc[cfg][lvl][i-1]>sc[cfg][lvl][i]) { // only increment if previous value is higher
                                if (sc[cfg][lvl][i]<(MAXCAL-1)) {
                                    sc[cfg][lvl][i]++;  cntH++;  //checks value before increment
                                    break;  //if incremented breaks loop and waits till next burst
                                }
                            }
                        }
                    }else {
                        if (rxBuffer[stOff+i]<lvlTrg){
                            if (sc[cfg][lvl][i-1]>sc[cfg][lvl][i]) { // only increment if previous value is higher
                                if (sc[cfg][lvl][i]<(MAXCAL-1)) sc[cfg][lvl][i]++; cntH++;   //checks value before increment
                            }
                        }
                    }
                }

             }//for


      }//if ft!=1

      //VERT calibration---------------------
      //skip if ft=2
      if ((ft!=2)&&(rxBuffer[60]!=0))  {
        lvlTrg= calTrg[cfg][1][lvl]; //set vert level target
        //find starting point of 1st valid measurement by looking for last valid measurement
        if (rxBuffer[64]==0) stOff=51;
        else if (rxBuffer[65]==0) stOff=52;
        else if (rxBuffer[66]==0) stOff=53;
        else if (rxBuffer[67]==0) stOff=54;
        else {stOff=55;}
        stOff++;

        for (i=0; i<=12; i++) {     //scan the 13 burst measurements
            if (rxBuffer[stOff+i]<lvlTrg) {
                //allow increase only if following 3 cycles are not too high
                if (i<=8){
                    if ((rxBuffer[stOff+i]<lvlTrg)&&(rxBuffer[stOff+i+1]<lvlTrg)&&(rxBuffer[stOff+i+2]<lvlTrg)) {
                        if (i==0){
                            if (lc[cfg][lvl][i]<(MAXCAL-2)) {
                                lc[cfg][lvl][i]++; lc[cfg][lvl][i]++; //checks value before increment
                                lc[cfg][lvl][i+1]++;  cntV++;  //do 2nd one slightly
                                break;  //if incremented breaks loop and waits till next burst
                            }
                        } else if (lc[cfg][lvl][i-1]>lc[cfg][lvl][i]) { // only increment if previous value is higher
                            if (lc[cfg][lvl][i]<(MAXCAL-1)) {
                                lc[cfg][lvl][i]++; cntV++;  //checks value before increment
                                break;  //if incremented breaks loop and waits till next burst
                            }
                        }
                    }
                }else {
                    if (rxBuffer[stOff+i]<lvlTrg){
                        if (lc[cfg][lvl][i-1]>lc[cfg][lvl][i]) { // only increment if previous value is higher
                            if (lc[cfg][lvl][i]<(MAXCAL-1)) lc[cfg][lvl][i]++; cntV++;  //checks value before increment
                        }
                    }
                }
            }

         }//for
      }//if ft!=2

      //perform single point phase calibration (but only for ft=3 or 4 :Rotary)
      if ((ft==3)||(ft==4)){
          signed char sph = (signed char)rxBuffer[20]; //interpret the char as a signed char
          if (sph>1){               //+1 deg before adjustment
              if (ph[cfg][lvl][0]>4) {  //range check
                  ph[cfg][lvl][0]--;  cntP++;
              }
          } else if (sph<-1) {      //-1 deg before adjustment
              if (ph[cfg][lvl][0]<(MAXCAL-1)) {  //range check
                  ph[cfg][lvl][0]++;  cntP++;
              }
          }
      }

    }//if fmode



}
